<?php

namespace App\Services;

use Google\Cloud\TextToSpeech\V1\AudioConfig;
use Google\Cloud\TextToSpeech\V1\AudioEncoding;
use Google\Cloud\TextToSpeech\V1\SsmlVoiceGender;
use Google\Cloud\TextToSpeech\V1\SynthesisInput;
use Google\Cloud\TextToSpeech\V1\TextToSpeechClient;
use Google\Cloud\TextToSpeech\V1\VoiceSelectionParams;
use Spatie\Backup\Helpers\Backup;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use App\Models\Voice;
use Exception;

class GCPTTSService 
{
    private $client;

    /**
     * Initialize GCP client
     */
    public function __construct()
{
    try {
        $credentialsPath = config('services.gcp.key_path');
        if ($credentialsPath) {
            $contents = file_get_contents($credentialsPath);

            if ($contents === false || empty($contents)) {
                throw new Exception("Failed to retrieve GCP credentials from the specified path.");
            }

            $decoded_credentials = json_decode($contents, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Failed to decode GCP credentials JSON: " . json_last_error_msg());
            }

            $this->client = new TextToSpeechClient([
                'credentials' => $decoded_credentials,
            ]);  
        } else {
            throw new Exception("GCP key path is not configured.");
        }

    } catch (Exception $e) {
        Log::error("GCP Text-to-Speech client initialization failed: " . $e->getMessage());
        return response()->json(["exception" => "Credentials are incorrect or missing. Please notify the support team."], 422);
    }
}

    /**
     * Synthesize text less than 5000 characters
     *
     * @return result link 
     */
    public function synthesizeSpeech(Voice $voice, $text, $format, $file_name)
    {   
        $text = preg_replace("/\&/", "&amp;", $text);
        $text = preg_replace("/(^|(?<=\s))<((?=\s)|$)/i", "&lt;", $text);
        $text = preg_replace("/(^|(?<=\s))>((?=\s)|$)/i", "&gt;", $text);

        $ssml_text = '<speak>' . $text . '</speak>'; 

        switch ($format) {
            case 'mp3':
                $audio_encoding = AudioEncoding::MP3;
                break;
            case 'wav':
                $audio_encoding = AudioEncoding::LINEAR16;
                break;
            case 'ogg':
                $audio_encoding = AudioEncoding::OGG_OPUS;
                break;
            default:
                $audio_encoding = AudioEncoding::MP3;
                break;
        }   

        $input_text = (new SynthesisInput())
                    ->setSsml($ssml_text);

        $input_voice = (new VoiceSelectionParams())
                    ->setLanguageCode($voice->language_code)
                    ->setName($voice->voice_id);

        $audio_config = (new AudioConfig())
                    ->setAudioEncoding($audio_encoding);

        if ($this->client) {
            try {
                $response = $this->client->synthesizeSpeech($input_text, $input_voice, $audio_config);
                $audio_stream = $response->getAudioContent();

                $backup = new Backup();
                $upload = $backup->download();
                if (!$upload['status']) { 
                    return false; 
                }

                Storage::disk('audio')->put($file_name, $audio_stream); 

                $data['result_url'] = Storage::url($file_name); 
                $data['name'] = $file_name;
                
                return $data;
            } catch (Exception $e) {
                Log::error("Synthesis failed: " . $e->getMessage());
                return response()->json(["exception" => "Failed to synthesize speech."], 500);
            }
        } else {
            Log::error("GCP Text-to-Speech client is not initialized.");
            return response()->json(["error" => "Text-to-Speech service is unavailable."], 500);
        }
    }
}
