<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Http\Request;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Auth;
use App\Services\Statistics\DavinciUsageService;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;
use App\Models\SubscriptionPlan;
use App\Models\User;


class UserController extends Controller
{
    use Notifiable;

    /**
     * Display a user profile.
     *
     * Get the authenticated user's profile.
     *
     * @OA\Get(
     *      path="/api/v1/user/profile",
     *      operationId="userProfile",
     *      tags={"User Profile"},
     *      summary="Get user profile",
     *      description="Get the profile of the authenticated user.",
     *      security={{ "passport": {} }},
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\JsonContent(
     *              type="object",
     *          ),
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthenticated",
     *      ),
     * )
    */
    public function index(Request $request)
    {                         
        return $request->user();         
    }


    /**
     * Get user usage statistics information
     *
     * @OA\Get(
     *      path="/api/v1/user/profile/statistics",
     *      operationId="userStatistics",
     *      tags={"User Profile"},
     *      summary="User usage statistics",
     *      description="Get current user usage statistics information",
     *      security={{ "passport": {} }},
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\JsonContent(
     *              type="object",
     *              @OA\Property(property="words", type="integer", description="Total words generated by the user"),
     *              @OA\Property(property="images", type="integer", description="Total images generated by the user"),
     *              @OA\Property(property="contents", type="integer", description="Total contents generated by the user"),
     *              @OA\Property(property="synthesized", type="integer", description="Total AI Voiceover synthesize tasks created by the user"),
     *              @OA\Property(property="transcribed", type="integer", description="Total AI Speech to text tasks created by the user"),
     *              @OA\Property(property="codes", type="integer", description="Total AI Codes created by the user"),
     *          ),
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthenticated",
     *      ),
     * )
    */
    public function statistics(Request $request)
    {                         
        $year = $request->input('year', date('Y'));
        $month = $request->input('month', date('m'));

        $davinci = new DavinciUsageService($month, $year);

        return response()->json([
            'words' => $davinci->userTotalWordsGenerated(),
            'images' => $davinci->userTotalImagesGenerated(),
            'contents' => $davinci->userTotalContentsGenerated(),
            'synthesized' => $davinci->userTotalSynthesizedText(),
            'transcribed' => $davinci->userTotalTranscribedAudio(),
            'codes' => $davinci->userTotalCodesCreated(),
        ], 200);
         
    }



    /**
     * Update user information
     *
     * @OA\Put(
     *      path="/api/v1/user/profile",
     *      operationId="userUpdate",
     *      tags={"User Profile"},
     *      summary="Update user information",
     *      description="Updates the user's profile information",
     *      security={{ "passport": {} }},
     *      @OA\RequestBody(
     *          required=true,
     *          description="User data",
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="name",
     *                      description="User's first name",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                     property="email",
     *                     description="User's email address, must be unique",
     *                     type="string",
     *                 ),
     *                  @OA\Property(
     *                      property="job_role",
     *                      description="User's job role",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="company",
     *                      description="User's company name",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="website",
     *                      description="User's company website",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="phone_number",
     *                      description="User's phone number",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="address",
     *                      description="User's address",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="city",
     *                      description="User's city name",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="postal_code",
     *                      description="User's postal code",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="country",
     *                      description="User's country",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="profile_photo",
     *                      description="User's avatar image file (allowed types: jpg, png; max size: 5MB)",
     *                      type="string",
     *                      format="binary"
     *                  ),
     *              ),
     *          ),
     *      ),
     *      @OA\Response(
     *          response=400,
     *          description="Bad request"
     *      ),
     *      @OA\Response(
     *          response=404,
     *          description="Resource Not Found"
     *      ),
     *      @OA\Response(
     *          response=201,
     *          description="User information was updated successfully",
     *          @OA\MediaType(mediaType="application/json")
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthorized: User not authenticated",
     *      ),
     *      @OA\Response(
     *          response=419,
     *          description="Validation error or unsupported file extension",
     *      ),
     * )
    */
    public function update(Request $request)
    {           
        $user = $request->user();

        if($request->name != null) {
            $user->name = $request->name;
        }

        if ($request->email != null) {
            $validated = Validator::make($request->all(), [
                'email' => ['required','string','email','max:255',Rule::unique('users')->ignore($user)],
            ]);
    
            if ($validated->fails()) {
                return response()->json(['error' => $validated->errors()], 422);
            }
        }

        if($request->job_role != null) {
            $user->job_role = $request->job_role;
        }

        if($request->company != null) {
            $user->company = $request->company;
        }

        if($request->website != null) {
            $user->website = $request->website;
        }

        if($request->phone_number != null) {
            $user->phone_number = $request->phone_number;
        }

        if($request->address != null) {
            $user->address = $request->address;
        }

        if($request->city != null) {
            $user->city = $request->city;
        }

        if($request->postal_code != null) {
            $user->postal_code = $request->postal_code;
        }

        if($request->country != null) {
            $user->country = $request->country;
        }
        
        if (request()->has('profile_photo')) {
            
            $image = request()->file('profile_photo');

            $name = Str::random(20);
            
            $folder = '/uploads/img/users/';
            
            $filePath = $folder . $name . '.' . $image->getClientOriginalExtension();

            $imageTypes = ['jpg', 'jpeg', 'png', 'webp', 'svg'];
            if (!in_array(Str::lower($image->getClientOriginalExtension()), $imageTypes)) {
                return response()->json(['error'=>__('Avatar image must be in png, jpeg, svg or webp formats')], 419);
            } else {
                $this->uploadImage($image, $folder, 'public', $name);

                $user->profile_photo_path = $filePath;
            }
        }

        $user->save();

        return response()->json(['message' => 'User information was updated successfully'], 201);
        
    }


    /**
     * Store user's personal API keys
     *
     * @OA\Post(
     *      path="/api/v1/user/profile/api",
     *      operationId="userAPIKeys",
     *      tags={"User Profile"},
     *      summary="Update user's personal API keys",
     *      description="Updates user's AI cloud vendor personal API keys",
     *      security={{ "passport": {} }},
     *      @OA\RequestBody(
     *          required=true,
     *          description="User data",
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="openai_key",
     *                      description="User's personal OpenAI API key",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                     property="claude_key",
     *                     description="User's personal Claude API key",
     *                     type="string",
     *                 ),
     *                  @OA\Property(
     *                      property="gemini_key",
     *                      description="User's personal Gemini API key",
     *                      type="string"
     *                  ),
     *                  @OA\Property(
     *                      property="sd_key",
     *                      description="User's personal SD API key",
     *                      type="string"
     *                  ),
     *              ),
     *          ),
     *      ),
     *      @OA\Response(
     *          response=400,
     *          description="Bad request"
     *      ),
     *      @OA\Response(
     *          response=404,
     *          description="Resource Not Found"
     *      ),
     *      @OA\Response(
     *          response=201,
     *          description="Personal api keys have been saved successfully",
     *          @OA\MediaType(mediaType="application/json")
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthorized: User not authenticated",
     *      ),
     * )
    */
    public function storeAPI(Request $request)
    {           
        $user = $request->user();

        if($request->openai_key != null) {
            $user->personal_openai_key = $request->openai_key;
        }

        if($request->claude_key != null) {
            $user->personal_claude_key = $request->claude_key;
        }

        if($request->gemini_key != null) {
            $user->personal_gemini_key = $request->gemini_key;
        }

        if($request->sd_key != null) {
            $user->personal_sd_key = $request->sd_key;
        }

        $user->save();
        return response()->json(['message' => 'Personal api keys have been saved successfully'], 201);
    }


    /**
     * Store user's custom referral id
     *
     * @OA\Post(
     *      path="/api/v1/user/profile/referral",
     *      operationId="userCustomReferral",
     *      tags={"User Profile"},
     *      summary="Set custom referral ID",
     *      description="Store user's custom referral id set by the user",
     *      security={{ "passport": {} }},
     *      @OA\RequestBody(
     *          required=true,
     *          description="User data",
     *          @OA\MediaType(
     *              mediaType="multipart/form-data",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="value",
     *                      description="New custom referral ID name set by the user",
     *                      type="string"
     *                  ),
     *              ),
     *          ),
     *      ),
     *      @OA\Response(
     *          response=400,
     *          description="Bad request"
     *      ),
     *      @OA\Response(
     *          response=404,
     *          description="Resource Not Found"
     *      ),
     *      @OA\Response(
     *          response=201,
     *          description="Custom referral ID saved",
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthorized: User not authenticated",
     *      ),
     *      @OA\Response(
     *          response=412,
     *          description="Precondition Failed",
     *      ),
     *      @OA\Response(
     *          response=419,
     *          description="Referral ID is in use already",
     *      ),
     * )
    */
    public function updateReferral(Request $request)
    {
        if($request->value == null) return response()->json(['error' => __('Referral id value missing.')], 412);

        $check = User::where('referral_id', $request->value)->first();

        if ($check) {
            return response()->json(['error' => __('This Referral ID is already in use by another user, please enter another one')], 419);
        } else {
            $user = $request->user();
            $user->referral_id = $request->value;
            $user->save();
        }

        return response()->json(['message' => __('Custom referral ID saved')], 201);
    }


    /**
     * Delete user account
     *
     * Get the authenticated user's profile and delete account.
     *
     * @OA\Delete(
     *      path="/api/v1/user/profile",
     *      operationId="userDelete",
     *      tags={"User Profile"},
     *      summary="Delete user account",
     *      description="Get the profile of the authenticated user and delete account.",
     *      security={{ "passport": {} }},
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\JsonContent(
     *              type="object",
     *          ),
     *      ),
     *      @OA\Response(
     *          response=401,
     *          description="Unauthenticated",
     *      ),
     * )
    */
    public function delete(Request $request)
    {   
        $user = $request->user();

        if(!$user) {
            return response()->json(['status' => false, 'message' => __('User not found')], 404);
        }

        Auth::guard('api')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        $user->delete();

        return response()->json(['status' => true, 'message' => __('User deleted successfully')], 200);
    }


    /**
     * Upload user profile image
     */
    public function uploadImage(UploadedFile $file, $folder = null, $disk = 'public', $filename = null)
    {
        $name = !is_null($filename) ? $filename : Str::random(25);

        $image = $file->storeAs($folder, $name .'.'. $file->getClientOriginalExtension(), $disk);

        return $image;
    }


    /**
     * Format storage space to readable format
     */
    private function formatSize($size, $precision = 2) { 
        $units = array('B', 'KB', 'MB', 'GB', 'TB'); 
    
        $size = max($size, 0); 
        $pow = floor(($size ? log($size) : 0) / log(1000)); 
        $pow = min($pow, count($units) - 1); 
        
        $size /= pow(1000, $pow);

        return round($size, $precision) . $units[$pow]; 
    }
}
